
#undef __KERNEL__	/* Makefile lazyness ;) */
#include <stdio.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <arpa/inet.h>

#include <asm/types.h>          /* For __uXX types */
#include <net/if.h>
#include <netinet/ip_icmp.h>
#include <netinet/udp.h>
#include <netinet/tcp.h>
#include <linux/ip_fw.h>        /* For IP_FW_MASQ_CTL */
#include <linux/ip_masq.h>      /* For specific masq defs */
#include <net/ip_masq.h>

#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <sys/param.h>

#define DEF_SCHED	"wlc"

int parse_addr(char *buf, u_int32_t *addr, u_int16_t *port);
void usage_exit(char **argv);
void fail(int err, char *text);
void list_forwarding_exit(void);

int main(int argc, char **argv)
{
	struct ip_masq_ctl mc;
	int cmd, c;
	int pares;
	char *optstr;
	int result;
	int sockfd;

	memset (&mc, 0, sizeof(struct ip_masq_ctl));
	/*
	 *	Want user virtual server control
	 */
	if ((cmd = getopt (argc, argv, "ADCadlLh")) == EOF) 
		usage_exit(argv);

	switch (cmd) {
	case 'A':	
		mc.m_cmd = IP_MASQ_CMD_ADD;
		mc.m_target = IP_MASQ_TARGET_VS;
		optstr = "t:u:s:";
		break;
	case 'D':
		mc.m_cmd = IP_MASQ_CMD_DEL;
		mc.m_target = IP_MASQ_TARGET_VS;
		optstr = "t:u:";
		break;
	case 'a':
		mc.m_cmd = IP_MASQ_CMD_ADD;
		mc.m_target = IP_MASQ_TARGET_VS_SCHED;
		optstr = "t:u:w:r:R:gmi";
		break;
	case 'd':
		mc.m_cmd = IP_MASQ_CMD_DEL;
		mc.m_target = IP_MASQ_TARGET_VS_SCHED;
		optstr = "t:u:w:r:R:";
		break;
	case 'C':
		mc.m_cmd = IP_MASQ_CMD_FLUSH;
		mc.m_target = IP_MASQ_TARGET_VS;
		optstr = "";
		break;
	case 'L':
	case 'l':
		list_forwarding_exit();
	default:
		usage_exit(argv);
	}

	/* use direct routing as default forwarding method */
	mc.u.vs.masq_flags = IP_MASQ_F_VS_DROUTE;
	
	while ((c = getopt (argc, argv, optstr)) != EOF) {
	        switch (c) {
		case 't':
		case 'u':
			if (mc.u.vs.protocol != 0)
				fail(2, "protocol already specified");
			mc.u.vs.protocol=(c=='t' ? IPPROTO_TCP : IPPROTO_UDP);
			pares = parse_addr(optarg, &mc.u.vs.vaddr, 
					   &mc.u.vs.vport);
			if (pares != 2) fail(2, "illegal virtual server "
					     "address:port specified");
			break;
		case 'i':
			mc.u.vs.masq_flags = IP_MASQ_F_VS_TUNNEL;
			break;
		case 'g':
			mc.u.vs.masq_flags = IP_MASQ_F_VS_DROUTE;
			break;
		case 'm':
			mc.u.vs.masq_flags = 0;
			break;
		case 'r':
		case 'R':
			pares = parse_addr(optarg, &mc.u.vs.daddr, 
					   &mc.u.vs.dport);
			if (pares == 0) fail(2, "illegal virtual server "
					     "address:port specified");
			/* copy vport to dport if none specified */
			if (pares == 1) mc.u.vs.dport = mc.u.vs.vport;
			break;
		case 'w':
			if (mc.u.vs.weight != 0)
				fail(2, "multiple server weights specified");
			if (sscanf(optarg, "%d", &mc.u.vs.weight) != 1)
				fail(2, "illegal weight specified");
			break;
		case 's':
		        if (strlen(mc.m_tname) != 0)
			        fail(2, "multiple scheduling modules specified");
			strncpy(mc.m_tname, optarg, IP_MASQ_TNAME_MAX);
			break;
		default:
			fail(2, "invalid option");
		}
	}

	if (optind < argc)
	  fail(2, "unknown arguments found in command line");

	/* set the default scheduling algorithm if none specified */
	if (mc.m_target == IP_MASQ_TARGET_VS && mc.m_cmd == IP_MASQ_CMD_ADD
	    && strlen(mc.m_tname) == 0) {
	        strcpy(mc.m_tname,DEF_SCHED);
	}
		
	sockfd = socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
	if (sockfd==-1) {
		perror("socket creation failed");
		exit(1);
	}

	result = setsockopt(sockfd, IPPROTO_IP, 
			    IP_FW_MASQ_CTL, (char *)&mc, sizeof(mc));
	if (result) {
		perror("setsockopt failed");
		/* print most common error messages */
		switch (cmd) {
		case 'A':	
			if (errno == EEXIST)
				printf("Service already exists\n");
			else if (errno == ESRCH)
				printf("Scheduler not found: ip_vs_%s.o\n",
				       mc.m_tname);
			break;
		case 'D':
			if (errno==ESRCH)
				printf("No such service\n");
			break;
		case 'a':
			if (errno == EEXIST)
				printf("Destination already exists\n");
			else if (errno == ESRCH)
				printf("Service not defined\n");
			break;
		case 'd':
			if (errno == ESRCH)
				printf("No such destination\n");
			break;
		}
	}
	return result;	
}


/* get ip address from the argument. 
 * Return 0 if failed,
 * 	  1 if addr read
 *        2 if addr and port read
 */
int parse_addr(char *buf, u_int32_t *addr, u_int16_t *port)
{
	char *pp;
	long prt;
	
	pp = strchr(buf,':');
	if (pp) *pp = '\0';

	*addr = inet_addr(buf);
	if (*addr == (u_int32_t)-1) return 0;
	if (pp == NULL) return 1;
	prt = atoi(++pp);
	if ((prt < 0) || (prt > 65535)) return 0;
	*port = htons(prt);
	return 2;
}

void usage_exit(char **argv) {
	char *p = argv[0];
	printf("\nUsage:\n"
	       "\t%s -A -[t|u] v.v.v.v:vport [-s scheduler]\n"
	       "\t%s -D -[t|u] v.v.v.v:vport\n"
	       "\t%s -C\n"
	       "\t%s -a -[t|u] v.v.v.v:vport -r d.d.d.d[:dport] [-g|-m|-i]"
	       "[-w wght]\n"
	       "\t%s -d -[t|u] v.v.v.v:vport -r d.d.d.d[:dport]\n"
	       "\t%s [-L|-l]\n",p,p,p,p,p,p);
	printf("\nCommands:\n"
	       "\t -A	add virtual service and link a scheduler to it\n"
	       "\t -D	delete virtual service\n"
	       "\t -C	clear the whole table\n"
	       "\t -a	add real server and select forwarding method\n"
	       "\t -d	delete real server\n"
	       "\t -L	list the table\n");
	printf("\nOptions:\n"
	       "\t protocol:	t :	tcp\n"
	       "\t		u :	udp\n"
	       "\t forwarding:	g :	gatewaying (routing) (default)\n"
	       "\t		m :	masquerading\n"
	       "\t		i :	ipip encapsulation (tunneling)\n"
	       "\t scheduler:	scheduling module 'ip_vs_<scheduler>.o'\n"
	       "\t 		default scheduler is '%s'.\n"
	       "\t weight:	capacity of the real server\n"
	       "\t addresses	v.v.v.v:vport	virtual (local) address\n"
	       "\t		d.d.d.d:dport	real server (destination) "
		"address\n", DEF_SCHED);
	exit(1);
}

void fail(int err, char *text) {
	printf("%s\n",text);
	exit(err);
}

void list_forwarding_exit(void)
{
    char buffer[1024];

    FILE *handle;
    handle = fopen ("/proc/net/ip_masq/vs", "r");
    if (!handle) {
	    printf ("Could not open /proc/net/ip_masq/vs\n");
	    printf ("Are you sure Virtual Server is supported by the kernel?\n");
	    exit (1);
    }
    while (!feof (handle)) {
	    if (fgets (buffer, sizeof(buffer), handle)) printf ("%s", buffer);
    }
    fclose (handle);
    exit (0);
}








