/*
 * Soft:        lvsgsp is simple connection tracking program for LVS
 *              <www.linuxvirtualserver.org>. It sum the global connections
 *              of a virtual server (ActiveConnection and InActiveConnection).
 *              Finaly it display the result to the stdout to be used with
 *              monitoring tools such as MRTG <www.mrtg.org>.
 *
 * Version:     $Id: main.c,v 0.0.4 2001/10/05 14:54:23 acassen Exp $
 *
 * Author:      Alexandre Cassen, <acassen@linux-vs.org>
 *
 *              This program is distributed in the hope that it will be useful,
 *              but WITHOUT ANY WARRANTY; without even the implied warranty of
 *              MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *              See the GNU General Public License for more details.
 *
 *              This program is free software; you can redistribute it and/or
 *              modify it under the terms of the GNU General Public License
 *              as published by the Free Software Foundation; either version
 *              2 of the License, or (at your option) any later version.
 */

#include "main.h"

/* Global vs var */
vserver vsinfo;

static void vs_filter(const char *buffer)
{
  /* virtual service variables */
  char protocol[10];
  static unsigned short vport;
  static u_int32_t fwmark;
  char scheduler[10];
  char flags[40];
  unsigned int timeout;

  /* destination variables */
  static char arrow[10];
  static struct in_addr ipaddr;
  unsigned short dport;
  static char fwd[10];
  int weight;
  int activeconns;
  int inactconns;
        
  int n;
  unsigned long temp;
  unsigned long temp2;

  if (buffer[0] == ' ') {
    /* parse destination entry */
    if ((n = sscanf(buffer, " %s %lX:%hX %s %d %d %d",
                            arrow, &temp, &dport, fwd, &weight,
                            &activeconns, &inactconns)) == -1)
      exit(1);

    ipaddr.s_addr = (uint32_t)htonl(temp);
#ifdef DEBUG
    printf("%s", buffer);
    printf("  SVR : %s [act(%d) inact(%d)]\n", inet_ntoa(ipaddr), activeconns, inactconns);
#endif

    if (vsinfo.vsfill) {
      if (vsinfo.rsip != 0) {
        if ((temp == vsinfo.rsip) && (dport == vsinfo.rsport)) {
          vsinfo.activeconns += activeconns;
          vsinfo.inactconns  += inactconns;
        }
      } else {
        vsinfo.activeconns += activeconns;
        vsinfo.inactconns  += inactconns;
      }
    }
  } else if (buffer[0] == 'F') {
    if ((n = sscanf(buffer, "%s %X %s %s %u %lX",
                            protocol, &fwmark, scheduler,
                            flags, &timeout, &temp2)) == -1)
      exit(1);
#ifdef DEBUG
    printf("FWMARK %d\n", fwmark);
#endif
    vsinfo.vsfill = (fwmark == vsinfo.fwmark)?1:0;
  } else {
    /*
     * Otherwise this is a TCP/UDP
     * virtual service entry.
     */
    if ((n = sscanf(buffer, "%s %lX:%hX %s %s %u %lX",
                            protocol, &temp, &vport, scheduler,
                            flags, &timeout, &temp2)) == -1)
      exit(1);

    ipaddr.s_addr = (uint32_t)htonl(temp);
#ifdef DEBUG
    printf("%s", buffer);
#endif
    if ((temp == vsinfo.ipaddr) && (vport == vsinfo.port))
      vsinfo.vsfill = 1;
    else
      vsinfo.vsfill = 0;

#ifdef DEBUG
    printf("VS : %s:%d\n", inet_ntoa(ipaddr), vport);
    ipaddr.s_addr = (uint32_t)htonl(vsinfo.ipaddr);
    printf("VS IPADDRESS : %s\n", inet_ntoa(ipaddr));
#endif
  }
}

static void list_vs()
{
  static char buffer[128];
  FILE *handle;
  int i;

  /* ready to check VS support */
  handle = fopen(VS_PROC_FILE, "r");
  if (!handle) {
    fprintf(stderr, "Could not open %s.\n"
                    "Are you sure that IP Virtual Server is supported "
                    "by the kernel ?\n", VS_PROC_FILE);
    exit(1);
  }

  /*
   * Read and print the first three head lines
   */
  for (i=0; i<3 && !feof(handle); i++)
    fgets(buffer, sizeof(buffer), handle);
        
  /*
   * Print the VS information according to the format
   */
  while (!feof(handle)) {
    if (fgets(buffer, sizeof(buffer), handle))
      vs_filter(buffer);
  }

  fclose(handle);
}

/* Usage display */
static void usage(const char *prog)
{
  fprintf(stderr, PROG" version "VERSION" Copyright(C) "AUTHOR"\n");
  fprintf(stderr, "Usage: %s [[-v vsipaddr vsport]|[-f fwmark]] [-r rsipaddr rsport]\n", prog);
  fprintf(stderr, "  -v vsipaddr vsport : ip address & port of the virtual server\n");
  fprintf(stderr, "  -f fwmark          : the fwmark number\n");
  fprintf(stderr, "  -r rsipaddr rsport : ip address & port of the real server\n");
  fprintf(stderr, "                       Belong to -v or -f VS\n");
  fprintf(stderr, "Samples:\n");
  fprintf(stderr, "  TCP|UDP VS: %s -v 192.168.200.10 80\n", prog);
  fprintf(stderr, "  TCP|UDP VS with RS: %s -v 192.168.200.10 80 -r 192.168.100.1 80\n", prog);
  fprintf(stderr, "  FWMARK VS: %s -f 1\n", prog);
}

/* parse command line  */
static void cmdline(int argc, char *argv[])
{
  int c;

  while (1) {
    c = getopt(argc, argv, "f:v:r:h");
    if (c == EOF) break;

    switch (c) {
      case 'f':
        if (argc < 3) goto error;
        vsinfo.fwmark = atoi(argv[2]);
        if (argc == 3) return;
        break;
      case 'v':
        if (argc < 4) goto error;
        /* fill the vsinfo struct */
        vsinfo.ipaddr = htonl(inet_addr(argv[2]));
        vsinfo.port   = atoi(argv[3]);
        if (argc == 4) return;
        break;
      case 'r':
        if (argc < 6) goto error;
        /* fill the vsinfo struct */
        vsinfo.rsip   = htonl(inet_addr(argv[argc-2]));
        vsinfo.rsport = atoi(argv[argc-1]);
        return;
      case 'h':
        goto error;
        break;
    }
  }

error:
  usage(argv[0]);
  exit(1);
}

/* Entry point */
int main(int argc, char **argv)
{
  /* clean our memory space */
  memset(&vsinfo, 0, sizeof(vserver));

  /* parse command line */
  cmdline(argc, argv);

  /* list the current VS topology */
  list_vs();

  /* finaly display the total number of active/inactive connections */
  printf("%d\n%d\ntime\n", vsinfo.activeconns, vsinfo.inactconns);

  exit(0);
}
